<?php

/**
 * Модель управления и синхронизации тегов

 * This is the model class for table "{{tag}}".
 *
 * The followings are the available columns in table '{{tag}}':
 * @property integer $id
 * @property string $tag
 * @property integer $frequency
 */
class Tag extends CActiveRecord
{

    /*
     * Разделитель между тегами
     */
    const RAZD = ', ';



    /**
     * Returns the static model of the specified AR class.
     * @return Tag the static model class
     */
    public static function model($className=__CLASS__)
    {
        return parent::model($className);
    }

    /**
     * @return string the associated database table name
     */
    public function tableName()
    {
        return '{{tag}}';
    }

    /**
     * @return array validation rules for model attributes.
     */
    public function rules()
    {
        // Правил нет, ибо юзер никогда сам не будет создавать/изменять/удалять эту модель
        return array();
    }

    /**
     * @return array relational rules.
     */
    public function relations()
    {
        // NOTE: you may need to adjust the relation name and the related
        // class name for the relations automatically generated below.
        return array(
        );
    }

    /**
     * @return array customized attribute labels (name=>label)
     */
    public function attributeLabels()
    {
        return array(
            'id' => 'ID',
            'tag' => 'Tag',
            'frequency' => 'Frequency',
        );
    }
    


    /*
     * Синхронизация тегов для НОВОЙ статьи
     */
    public function createPostTags ($tags)
    {
        // Если тегов нет, то explode возвратит не пустой массив, а массив с одним элементом - пустой строкой
        // Преабразуем его в нормальный пустой массив
        if (empty ($tags["0"])) $tags = array ();
        
            foreach ($tags as $value)
            {
                $tag = $this->find('`name` = :tag', array(':tag' => $value));

                // Если тега не существует, создаем его
                if ($tag === NULL)
                {
                    $new_tag = new Tag;
                    $new_tag->name = $value;
                    $new_tag->frequency = 1;
                    $new_tag->save();
                }
                // Если существует, увеличиваем встречаемость
                else
                {
                    $tag->frequency++;
                    $tag->save();
                }
            }
    }

    /*
     * Синхронизация тегов для ОБНОВЛЯЕМОЙ статьи
     */
    public function updatePostTags ($tags_old, $tags_new)
    {
        // Если тегов нет, то explode возвратит не пустой массив, а массив с одним элементом - пустой строкой
        // Преабразуем его в нормальный пустой массив
        if (empty ($tags_old["0"])) $tags_old = array ();
        if (empty ($tags_new["0"])) $tags_new = array ();

            // 1. Выбираем добавленные теги
            $result_new = array_diff($tags_new, $tags_old);

            foreach ($result_new as $value)
            {
                $tag = $this->find('`name` = :tag', array(':tag' => $value));
                // Если тега не существует, создаем его
                if ($tag === NULL)
                {
                    $new_tag = new Tag;
                    $new_tag->name = $value;
                    $new_tag->frequency = 1;
                    $new_tag->save();
                }
                // Если существует, увеличиваем встречаемость
                else
                {
                    $tag->frequency++;
                    $tag->save();
                }
            }

            // 2. Выбираем убранные теги
            $result_old = array_diff($tags_old, $tags_new);

            foreach ($result_old as $value)
            {
                // Уменьшаем встречаемость
                $tag = $this->find('`name` = :tag', array(':tag' => $value));

                $tag->frequency--;
                $tag->save();

                // Если встречаемость упала до нуля, удаляем тег
                if ($tag->frequency == 0)
                {
                    $tag->delete();
                }
            }
    }

    /*
     * Синхронизация тегов для УДАЛЯЕМОЙ статьи
     */
    public function deletePostTags ($tags)
    {
        // Если тегов нет, то explode возвратит не пустой массив, а массив с одним элементом - пустой строкой
        // Преабразуем его в нормальный пустой массив
        if (empty ($tags["0"])) $tags = array ();

            // Проходим по ВСЕМ тегам удаляемой статьи
            foreach ($tags as $value)
            {
                // Уменьшаем встречаемость
                $tag = $this->find('`name` = :tag', array(':tag' => $value));

                $tag->frequency--;
                $tag->save();

                // Если встречаемость упала до нуля, удаляем тег
                if ($tag->frequency == 0)
                {
                    $tag->delete();
                }
            }
    }

    /*
     * ПОЛНАЯ синхронизация тегов.
     *
     * Происходит очистка таблицы тегов и новое ее заполнение путем анализа тегов в статьях.
     * Может занимать продолжительное время.
     */
    public function syncAllTags ()
    {
        // Выбираем ВСЕ статьи
        $array_post = Post::model ()->findAll ();

        // Очищаем таблицу тегов
        Yii::app ()->db->createCommand ('TRUNCATE ' . $this->tableName ())->execute ();

        foreach ($array_post as $post)
        {
            // Проводим с тегами те же действия, что и при добавлении новой статьи
            $this->createPostTags (explode (self::RAZD, $post->tags));
        }
    }



    /**
     * Загрузка тегов для облака на главной
     * @return CActiveDataProvider the data provider that can return the models based on the search/filter conditions.
     */
    public function loadTags ()
    {
        $criteria = new CDbCriteria (array (
            'order' => '`frequency` DESC'
        ));
        return new CActiveDataProvider (get_class ($this), array (
            'criteria' => $criteria,
            'pagination' => array (
                'pageSize' => Yii::app ()->config->tags_cloud_links_count
            )
        ));
    }

    /**
     * Загрузка всех тегов
     * @return CActiveDataProvider the data provider that can return the models based on the search/filter conditions.
     */
    public function loadAllTags ()
    {
        $criteria = new CDbCriteria (array (
            'order' => '`frequency` DESC'
        ));
        return new CActiveDataProvider (get_class ($this), array (
            'criteria' => $criteria,
            'pagination' => array (
                'pageSize' => 1000000
            )
        ));
    }

}